/*********************************************************************
*	 Project: MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=32MHz, SysCLK=32MHz
*	 Description:
*			PTM0 clock:	96MHz
*			CH0(P22)/CH1(P33),CH2(P24)/CH3(P34),CH4(P26)/CH5(P35)
*			6 Channel, 3 pairs of COMP,16bit,Central Aligned,Dead Time
*			PWM	Freq: 96MHz/(2048*2)=23.437KHz
*	 Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		32000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26



#define	PWM_MIN			(0*2)
#define	PWM_MAX			(1024*2)
#define	PWM_3_4			(768*2)
#define	PWM_2_4			(512*2)
#define	PWM_1_4			(256*2)
#define	PWM_LOW			(62*2)
#define	PWM_HINT		(64*2)

#define	PTM0_RELOAD		(PWM_MAX)

#define	PTM0_C			 (65536)

#define	PTM0_CL(x)		(u8)((~(u16)x)+1)
#define	PTM0_CH(x)		(u8)(((~(u16)x)+1)>>8)

idata WordTypeDef wDuty[3];
u8 uDutyTimecnt;

/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT6);						// set P26 as push-pull(LED)
	PORT_SetP2PushPull(BIT2|BIT3|BIT4);				// set P22(CEX0),P23(CEX2),P24(CEX4) as	push-pull for PWM output
	PORT_SetP3PushPull(BIT2|BIT4|BIT5);				// set P32(CEX1),P34(CEX3),P35(CEX5) as	push-pull for PWM output
}


/***********************************************************************************
*Function:		void InitPTM0_PWM(void)
*Description:	Initialize PTM0	for	PWM
*Input:
*Output:
*************************************************************************************/
void InitPTM0_PWM(void)
{
	PTM0_SetCLOCK_CKMIX16();			// 96MHz

	PTM0_CH0_SetMode_PWM();
	PTM0_CH1_SetMode_PWM();
	PTM0_CH2_SetMode_PWM();
	PTM0_CH3_SetMode_PWM();
	PTM0_CH4_SetMode_PWM();
	PTM0_CH5_SetMode_PWM();

	PTM0_CH0_SetPWM_16Bit();
	PTM0_CH1_SetPWM_16Bit();
	PTM0_CH2_SetPWM_16Bit();
	PTM0_CH3_SetPWM_16Bit();
	PTM0_CH4_SetPWM_16Bit();
	PTM0_CH5_SetPWM_16Bit();

	PTM0_CH01_SetPWM_EnBufferMode();		// CH0,CH1 Enable Buffer mode
	PTM0_CH23_SetPWM_EnBufferMode();		// CH2,CH3 Enable Buffer mode
	PTM0_CH45_SetPWM_EnBufferMode();		// CH4,CH5 Enable Buffer mode
	PTM0_CH01_SetPWM_EnDeadTime();		// CH0,CH1 Enable Dead-Time
	PTM0_CH23_SetPWM_EnDeadTime();		// CH2,CH3 Enable Dead-Time
	PTM0_CH45_SetPWM_EnDeadTime();		// CH2,CH3 Enable Dead-Time

	PTM0_SetPWM_CenterAligned();			// Central Aligned
	PTM0_SetCenterAligned_CFonTop();		// CF on bottom	of PWM cycle

	PTM0_SetDeadTimePreScaler_SysclkDiv8();	// pre-scaler SysClk/8
	PTM0_SetDeadTimePeriod(4);				// period 4*(1/PreScaler)=4*(1/(Sysclk/8))=32/Sysclk

	PTM0_SetCounter(PTM0_C-PTM0_RELOAD);
	PTM0_SetCounterReload(PTM0_C-PTM0_RELOAD);

	// Set PWM duty
	PTM0_CH0_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH1_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH2_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH3_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH4_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH5_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));

	// Enable PWM output
	PTM0_SetPWM0_EnOutput();
	PTM0_SetPWM1_EnOutput();
	PTM0_SetPWM2_EnOutput();
	PTM0_SetPWM3_EnOutput();
	PTM0_SetPWM4_EnOutput();
	PTM0_SetPWM5_EnOutput();

	// Set PWM Output pin
	PTM0_SetCEXn_Pin(CEX0_P22|CEX1_P23|CEX2_P24|CEX3_P34|CEX4_P32|CEX5_P35);

	PTM0_CF_DisInterrupt();				// Disable PTM0	CF interrupt

	PTM0_EnPCACounter();					// Enable PTM0 counter

}


/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{
	InitClock();
	InitPort();
	InitPTM0_PWM();
}


void main()
{
	WordTypeDef	duty;

	InitSystem();

	IO_LED_G=0;
	DelayXms(1000);
	IO_LED_G=1;

	wDuty[0].W=PWM_MIN;
	wDuty[1].W=PWM_2_4;
	wDuty[2].W=PWM_MAX;


	while(1)
	{
		DelayXms(10);

		uDutyTimecnt++;
		if(uDutyTimecnt>=20)
		{
			uDutyTimecnt=0;
			IO_LED_G=!IO_LED_G;
		}

		if(wDuty[0].W>=PWM_MAX)
		{
			wDuty[0].W=PWM_MIN;
		}
		else
		{
			wDuty[0].W=wDuty[0].W+16;
		}
		if(wDuty[1].W>=PWM_MAX)
		{
			wDuty[1].W=PWM_MIN;
		}
		else
		{
			wDuty[1].W=wDuty[1].W+16;
		}
		if(wDuty[2].W>=PWM_MAX)
		{
			wDuty[2].W=PWM_MIN;
		}
		else
		{
			wDuty[2].W=wDuty[2].W+16;
		}
		duty.W=PTM0_C-wDuty[0].W;
		// Buffer mode,CH0/CH1 is reload by	CH1
		PTM0_CH1_SetValue(duty.B.BHigh,duty.B.BLow);
		duty.W=PTM0_C-wDuty[1].W;
		// Buffer mode,CH2/CH3 is reload by	CH3
		PTM0_CH3_SetValue(duty.B.BHigh,duty.B.BLow);
		duty.W=PTM0_C-wDuty[2].W;
		// Buffer mode,CH4/CH5 is reload by	CH5
		PTM0_CH5_SetValue(duty.B.BHigh,duty.B.BLow);
	}
}

